import time
import threading
import netifaces

from functools import wraps

def get_local_adresses():
	
	local_adresses = set()

	for interface in netifaces.interfaces():

		for adress in netifaces.ifaddresses(interface).get(netifaces.AF_INET, []):
			address = adress.get("addr", None)
			if address: local_adresses.add(address)

		for adress in netifaces.ifaddresses(interface).get(netifaces.AF_INET6, []):
			address = adress.get("addr", "").split("%")[0]
			if address: local_adresses.add(address)

	return list(local_adresses)


class memoize(object):
	"""Memoize With Timeout"""
	_caches = {}
	_timeouts = {}
	 
	def __init__(self, timeout=10):
		self.timeout = timeout
		 
	def collect(self):
		"""Clear cache of results which have timed out"""
		for func in self._caches:
			cache = {}
			for key in self._caches[func]:
				if (time.time() - self._caches[func][key][1]) < self._timeouts[func]:
					cache[key] = self._caches[func][key]
			self._caches[func] = cache
	 
	def __call__(self, f):
		self.cache = self._caches[f] = {}
		self._timeouts[f] = self.timeout
		 
		def func(*args, **kwargs):
			kw = kwargs.items()
			kw.sort()
			key = (args, tuple(kw))
			try:
				v = self.cache[key]
				if (time.time() - v[1]) > self.timeout:
					raise KeyError
			except KeyError:
				v = self.cache[key] = f(*args,**kwargs),time.time()
			return v[0]
		func.func_name = f.func_name
		 
		return func

def rate_limited(max_per_second, mode="wait", delay_first_call=False):
	"""
	Decorator that make functions not be called faster than

	set mode to "kill" to just ignore requests that are faster than the 
	rate.

	set delay_first_call to True to delay the first call as well
	"""
	lock = threading.Lock()
	min_interval = 1.0 / float(max_per_second)
	def decorate(func):
		last_time_called = [0.0]
		@wraps(func)
		def rate_limited_function(*args, **kwargs):
			def run_func():
				lock.release()
				ret = func(*args, **kwargs)
				last_time_called[0] = time.perf_counter()
				return ret
			lock.acquire()
			elapsed = time.perf_counter() - last_time_called[0]
			left_to_wait = min_interval - elapsed
			if delay_first_call:    
				if left_to_wait > 0:
					if mode == "wait":
						time.sleep(left_to_wait)
						return run_func()
					elif mode == "kill":
						lock.release()
						return
				else:
					return run_func()
			else:
				# Allows the first call to not have to wait
				if not last_time_called[0] or elapsed > min_interval:   
					return run_func()       
				elif left_to_wait > 0:
					if mode == "wait":
						time.sleep(left_to_wait)
						return run_func()
					elif mode == "kill":
						lock.release()
						return
		return rate_limited_function
	return decorate