{bridge}                = require "./../Bridge"
{ViewerTools}           = require "./../tools/ViewerTools"
{log}                   = require "./../tools/Log"
{HighlightComponent}    = require "./../highlight/HighlightComponent"
{EditStateController}   = require "./../edit/EditStateController"
{EditStates}            = require "./../edit/EditStates"
{Transform}             = require "./../tools/Transform"
require "./Styles"

class exports.StateController

    constructor: (@deviceContext) ->

        log "Setting up state controller", @deviceContext

        @selecting = false
        @dragging = false
        @clientLayers = []
        @eventLayers = []

        @directSelectionAllowed = true

        @state = {
            mode: "Default"
            codeState: "OK"
            selecting: false
            highlightingTarget: {}
            editingTarget: {}
        }

        # Layer selection highlighting
        @highlighter = new HighlightComponent()

        # Layer editing
        @editor = new EditStateController()

        # Listen for state changes:
        bridge.on "studio:state", (state) =>

            log "StateController:state handler;", state

            switch state.mode
                when "Editing"      then @stopSelecting()
                when "Selecting"    then @startSelecting()  # Direct selection is disabled for now.
                when "Highlighting" then @stopSelecting()
                else                     @stopSelecting()

            @renderState state

        bridge.on "codePanel:delayedDeactivation", (value) =>
          if value?
            @delayStateChange = true
            if value == "propertiesPanel"
              @editor.updateState(EditStates.hidden)
          else
            @delayStateChange = false
            @renderState @delayedState

        bridge.on "drag:start", (data) =>
          @editor.animationTool.dragStart(data)

        bridge.on "drag:end", (data) =>
          @editor.animationTool.dragEnd(data)

        bridge.on "animationPanel:propertyChanged", (data) =>
          value = @coerceValue(data, _.first(@editor.animationTool.layers))
          @editor.animationTool.animationPanelPropertyChanged(data.name,value)

        # Listen for layer changes from the properties panel:
        bridge.on "propertiesPanel:propertyChanged", (data) =>
            layer = ViewerTools.firstLayerForTarget(@state.editingTarget)
            return if not layer?

            coercedValue = @coerceValue(data, layer)
            fromValues =
              "#{data.name}": layer[data.name]
            layer[data.name] = coercedValue
            toValues =
              "#{data.name}": layer[data.name]
            @editor.animationTool.updateProperties(fromValues,toValues)
            @editor.editLayer layer
            if @editor.relatedLayers?
              for layer in @editor.relatedLayers
                layer[data.name] = coercedValue

        # Listen for the device context being reset, and clean up lingering
        # edit targets:
        @deviceContext.on "reset", =>
            log "device context reset - StateController"

            if @state?.highlightingTarget or @state?.editingTarget
                @state?.editingTarget = null
                @state?.highlightingTarget = null
                @renderState @state

        # We may be reset while frozen. This leads to trouble when we are trying
        # to unfreeze, because listeners might be put back on layers that longer
        # exists. Runtime therefore sends out a "willReset" via the context, before
        # calling "reset" on the context. We use this to unfreeze (exit selectmode)
        # prior to that:
        @deviceContext.on "willReset", =>
            @stopSelecting()


    coerceValue: (data, layer) ->
        if not data.value? or data.value is ""
            defaultValue = layer.states.default[data.name]
            # Don't know what this was in for
            # if not defaultValue?
            #     if data.type is "number" or data.type is "@Number"
            #         defaultValue = 0
            return defaultValue
        else if data.type is "number" or data.type is "@Number"
            return parseFloat(data.value)
        else
            return ViewerTools.removeQuotes(data.value)

    removeQuotesFromProperties: (props) ->
      result = {}
      for key,value of props
        if _.isObject value
          value = @removeQuotesFromProperties(value)
        else
          value = ViewerTools.removeQuotes(value)
        result[key] = value
      result

    initializeAnimationEditing: (editingTarget) ->
      layer = ViewerTools.firstLayerForTarget(editingTarget)
      return unless layer?
      if not @editor.animationTool.isActive()
        @updateAnimationToolProperties(layer, editingTarget)
        @editor.snappingVisualizer.hideOutline()
        @editor.animationTool.target = editingTarget
        @editor.updateState(EditStates.animation)

    exitAnimationEditing: ->
        if @editor.state == EditStates.animation
          @editor.updateState(EditStates.idle)

    updateAnimationToolProperties: (layer, editingTarget) ->
      editingTargetLayerState = editingTarget?.state || "default"
      if editingTargetLayerState == "default" && editingTarget?
        newProps = _.clone(_.omit(editingTarget.propertyValues,editingTarget.frozenMembers))
        newProps = @removeQuotesFromProperties(newProps)
        oldProps = _.pick(layer,_.keys(newProps))
      else
        newProps = _.clone(layer.states[editingTargetLayerState])
        newProps.options = _.clone(_.omit(editingTarget.propertyValues,editingTarget.frozenMembers))
        newProps = @removeQuotesFromProperties(newProps)
        oldProps = _.pick(layer,_.keys(layer.states[editingTargetLayerState]))
      oldProps = @removeQuotesFromProperties(oldProps)
      @editor.animationTool.updateProperties(oldProps,newProps)
      return newProps

    # Render
    #

    renderState: (newState) ->
        if @delayStateChange
          @delayedState = newState
          return
        else
          @delayedState = null
        highlightingLayerId = newState.highlightingTarget?.runtimeId
        currentHighlightingLayerId = @state.highlightingTarget?.runtimeId

        editingTarget = newState.editingTarget
        editingTargetLayerHash = _.first(editingTarget?.relatedHashes) ? editingTarget?.owner
        editingTargetLayerState = editingTarget?.state || "default"
        editingTargetFrozenMembers = editingTarget?.frozenMembers
        editingPropertyType = editingTarget?.propertyType

        currentEditingTargetLayerHash = @state.editingTarget?.owner
        currentEditingPropertyType = @state.editingTarget?.propertyType
        log editingPropertyType,currentEditingPropertyType

        # Switch in the 'arrow' cursor when we're doing direct manipulation:
        if newState.selecting or editingTargetLayerHash? or highlightingLayerId?
            document.body.classList.add("editMode")
        else
            document.body.classList.remove("editMode")

        # Apply the correct highlighting:
        if highlightingLayerId isnt currentHighlightingLayerId
            if highlightingLayerId? and not editingTargetLayerHash?
                @highlighter.highlight(
                    ViewerTools.getLayerById(highlightingLayerId),
                    not newState.editingTarget?.owner?
                )
            else
                @highlighter.unhighlight()
        else
            @highlighter.unhighlight()
        if editingTarget?.expectedType in ["@AnimateParameters","@AnimationOptions"] && editingPropertyType == "Animation"
          @initializeAnimationEditing(editingTarget)
        else
          @exitAnimationEditing()
          # Apply the correct editing chrome:
          if editingTargetLayerHash isnt currentEditingTargetLayerHash or (editingPropertyType? and currentEditingPropertyType? and editingPropertyType isnt currentEditingPropertyType)
              relatedLayers = ViewerTools.getLayersForTarget(editingTarget)
              layer = _.first(relatedLayers)
              if layer? && layer.states.current.name != editingTargetLayerState

                  newProps = @updateAnimationToolProperties(layer, editingTarget)
                  if editingTargetLayerState == "default" && editingTarget?
                    for relatedLayer in relatedLayers
                      relatedLayer.props = newProps
                  else
                    for relatedLayer in relatedLayers
                        relatedLayer.stateSwitch(editingTargetLayerState)

              @editor.propertyLock.frozenMembers = null
              frozenRelatedMembers = ViewerTools.distinctPropertiesOfLayers(relatedLayers)
              if frozenRelatedMembers?
                bridge.send "framerView:lockProperties",
                  id:             layer?.id
                  hash:           editingTargetLayerHash
                  properties:     frozenRelatedMembers
              if editingTargetFrozenMembers? and frozenRelatedMembers?
                @editor.propertyLock.frozenMembers = editingTargetFrozenMembers.concat(frozenRelatedMembers)
              else if editingTargetFrozenMembers?
                @editor.propertyLock.frozenMembers = editingTargetFrozenMembers
              else if frozenRelatedMembers?
                @editor.propertyLock.frozenMembers = frozenRelatedMembers

              @editor.editTarget(editingTarget)

        @state = newState

    # Context wide
    #

    handleContextMouseDown: (event) =>

        return unless @state.selecting

        event.preventDefault()
        event.stopImmediatePropagation()

        target = event.target
        while target && !target.classList.contains("framerLayer")
            target = target.parent

        if target? and (layer = target.__framerInstance)?

            @mouseDownLayer = layer

            document.body.addEventListener "mouseup", @handleContextMouseUp, true

            @mouseDownFrame = Transform.getLayerScreenFrame(layer)
            @mouseDownX = event.screenX
            @mouseDownY = event.screenY

    handleContextMouseUp: (event) =>

        return unless @state.mode is "Selecting"

        document.body.removeEventListener "mouseup", @handleContextMouseUp, true

        event.preventDefault()
        event.stopImmediatePropagation()

        if @directSelectionAllowed and @mouseDownLayer?
            bridge.send "framerView:layerClick",
                hash: @mouseDownLayer.__framerInstanceInfo?.hash
                state: @mouseDownLayer.states.current.name

        @mouseDownLayer = null if @mouseDownLayer?

    handleContextMove: (event) =>

        return unless @state.selecting

        # When the left mouse button is down, then we're dragging: handle this as
        # a dragging event:
        return if event.which is 1

        # Mouse move is triggered on CMD up. This means we're escaping
        # selection mode but that the FS state hasn't updated yet. If
        # that's the case, ignore this event:
        return unless event.metaKey

        # Ignore mouse over's when in 'selecting' mode, and direct select is "off":
        # return if not @directSelectionAllowed and @state.mode is "Editing"

        event.preventDefault()
        event.stopImmediatePropagation()

        # Traverse the DOM tree from the target node up to the first div that is
        # a "framerLayer":
        target = event.target
        while target && !target.classList.contains("framerLayer")
            target = target.parent

        if (target? and
            (layer = target.__framerInstance)?
        )
            # if layer.id isnt @state.highlightingTarget.runtimeId
            bridge.send "framerView:layerHighlighted",
                id: layer.id
                hash: layer.__framerInstanceInfo?.hash
                state: layer.states.current.name
        else
            bridge.send "framerView:unhighlightLayer"


    # Select Mode
    #

    startSelecting: ->
        return if @selecting or not @editor.mouse?
        @selecting = true

        context = @deviceContext

        @clientLayers = context._layers.concat() # copy

        for layer in @clientLayers
            if layer.ignoreEvents
              layer.ignoreEvents = false
            else
              @eventLayers.push(layer)
            layer._element.__framerInstance = layer

        document.body.addEventListener "mousemove", @handleContextMove, true
        document.body.addEventListener "mousedown", @handleContextMouseDown, true

    stopSelecting: ->
        return unless @selecting

        context = @deviceContext

        document.body.removeEventListener "mousedown", @handleContextMouseDown, true
        document.body.removeEventListener "mousemove", @handleContextMove, true

        for layer in @clientLayers
            if layer not in @eventLayers
              layer.ignoreEvents = true
            delete layer._element.__framerInstance

        @clientLayers = []
        @eventLayers = []
        @highlighter.unhighlight()
        @selecting = false

    # Utils
    #

    log: (args...) ->
        # console.log args...
