{bridge}                    = require "./../Bridge"
{log}                       = require "./../tools/Log"
{ViewerTools}               = require "./../tools/ViewerTools"
{EditStates}                = require "./EditStates"

{TranslationTool}           = require "./tools/TranslationTool"
{ResizingTool}              = require "./tools/ResizingTool"
{RotationTool}              = require "./tools/RotationTool"
{BorderRadiusTool}          = require "./tools/BorderRadiusTool"
{AnimationTool}             = require "./tools/AnimationTool"

{TransformController}       = require "./transform/TransformController"

{PropertyLockController}    = require "./PropertyLockController"

{SnappingModel}             = require "./../snapping/SnappingModel"
{SnappingVisualizer}        = require './../snapping/SnappingVisualizer'

{ViewController}            = require "./view/ViewController"
{CursorController}          = require "./view/CursorController"
{DragHandle}                = require "./view/DragHandle"

{Shapes}                    = require './../snapping/Shapes'

{TooltipManager}            = require "./tooltip/TooltipManager"

class exports.EditStateController extends Framer.BaseClass

    shapes = new Shapes()

    constructor: (options) ->

        @state =                    EditStates.idle

        @snappingModel =            new SnappingModel
        @snappingVisualizer =       new SnappingVisualizer

        @translationTool =          new TranslationTool
        @resizingTool =             new ResizingTool
        @rotationTool =             new RotationTool
        @borderRadiusTool =         new BorderRadiusTool
        @animationTool =            new AnimationTool

        @transformController =      new TransformController

        @cursorController =         new CursorController
        @cursorController.editor = @
        @tooltip =                  new TooltipManager

        @resizeHandles =            @configureHandles()
        @animationControlActions =
          clickedPlay: @animationTool.clickedPlay
          clickedLooping: @animationTool.clickedLooping
          clickedSlomo: @animationTool.clickedSlomo

        @viewController =           new ViewController(@, @resizeHandles,@animationControlActions)

        @propertyLock =             new PropertyLockController

        window.addEventListener "mousemove", @handleMouseMove, true
        window.addEventListener "mousedown", @handleMouseDown, true
        window.addEventListener "mouseup", @handleMouseUp, true
        window.addEventListener "keydown", @handleKeyDown, true
        window.addEventListener "keyup", @handleKeyUp, true
        bridge.on "studio:framerViewLeave", @handleMouseOut, true

        window.addEventListener "resize", @updateLayerInfo, true

        @sendThrottledToBridge = _.throttle @sendToBrige, 1000 / 30

    configureHandles: =>
        dragHandles = []
        for x in [0, 0.5, 1]
            for y in [0, 0.5, 1]
                dragHandle = new DragHandle(fx:x, fy:y, index:dragHandles.length)
                dragHandle.handles = dragHandles
                dragHandles.push(dragHandle)
        return dragHandles

    # interaction

    handleMouseDown: (event) =>
        return unless event.which is 1
        return @animationTool.handleMouseDown(event, @) if @state is EditStates.animation
        return unless @layer?
        event.preventDefault()
        event.stopImmediatePropagation()

        bridge.send "drag:start"
        return @transformController.handleMouseDown(event, @) if @state is EditStates.transform
        return unless @state is EditStates.idle or EditStates.locked

        @mouse = {x:event.clientX, y:event.clientY}

        newState = @stateForPosition(@mouse)
        @updateState(newState)
        @tooltip.showInfo(@)

    handleMouseUp: (event) =>
        return unless event.which is 1
        return @animationTool.handleMouseUp(event, @) if @state is EditStates.animation
        return unless @layer?
        event.preventDefault()
        event.stopImmediatePropagation()
        @sendThrottledToBridge.cancel()
        # @sendToBrige()
        bridge.send "drag:end"
        @tooltip.hide()
        return if @state is EditStates.idle
        return @transformController.handleMouseUp(event, @) if @state is EditStates.transform


        @updateState(EditStates.idle)
        @updateLayerInfo()
        @stateForPosition(@mouse)

    handleMouseMove: (event) =>
        beforeChange = _.pick(@layer,@propertyLock.activeProperties())
        @mouse = {x:event.clientX, y:event.clientY}
        return @animationTool.handleMouseMove(event, @mouse, @) if @state is EditStates.animation
        return unless @layer?
        event.preventDefault()
        event.stopImmediatePropagation()

        return @updateState(EditStates.transform) if @state is EditStates.idle and event.metaKey

        controller = @controllerForState()
        if controller?
            controller.handleMouseMove?(event, @)
            @snappingVisualizer.showLayerOutline(@layer)
            @tooltip.showInfo(@)
            @cursorController.show(@)
            if event.which is 1
                if controller.layerChanged?
                    @layerChanged(beforeChange) if controller.layerChanged()
                else
                    @layerChanged(beforeChange)
            return

        @stateForPosition(@mouse)
        @viewController.render()

    handleMouseOut: (event) =>
        @mouse = null
        return unless @layer?
        return if @state is EditStates.animation
        @controllerForState()?.handleMouseOut?(event, @)

    # keyboard

    handleKeyDown: (event) =>
        return unless @layer?
        return if @state is EditStates.animation
        @controllerForState()?.handleKeyDown?(event, @)

        if @state is EditStates.idle or @state is EditStates.locked
            if event.metaKey
                @updateState(EditStates.transform) if @mouse?
            else
                @translationTool.handleKeyDown(event, @)

    handleKeyUp: (event) =>
        return unless @layer?
        return if @state is EditStates.animation
        @controllerForState()?.handleKeyUp?(event, @)
        event.preventDefault()
        event.stopImmediatePropagation()

    # state

    stateForPosition: (mouse) ->
        return unless mouse
        return if @state is EditStates.animation
        for handle in @resizeHandles when handle.isActive
            if handle.inDragDistance(mouse)
                if @propertyLock.resizingHandle(handle)
                    @cursorController.showLock()
                    return EditStates.locked
                @activeHandle = handle
                @cursorController.showResizingCursor(@centerPoint(), handle)
                return EditStates.resizing

        if @viewController.borderRadiusHandle?.inDragDistance(mouse)
            if @propertyLock.borderRadius()
                @cursorController.showLock()
                return EditStates.locked
            @cursorController.remove()
            return EditStates.borderRadius

        layerPolyline = @layerPolyline()
        mouseInsideLayer = shapes.pointInside(layerPolyline, mouse) if @layer.scale isnt 0
        if mouseInsideLayer
            if @propertyLock.translation()
                @cursorController.showLock()
                return EditStates.locked
            @cursorController.remove()
            return EditStates.moving

        # don't show rotate layer if mouse is too close near the layer
        if @layer.width > 0 and @layer.height > 0
            offsetLayer = shapes.offsetPolyline(layerPolyline, 8)
            mouseInsideLayer = shapes.pointInside(offsetLayer, mouse) if @layer.scale isnt 0
            if mouseInsideLayer
                @cursorController.remove()
                return EditStates.idle

        for handle in @resizeHandles when handle.isActive
            if handle.inRotateDistance()
                if @propertyLock.rotation()
                    @cursorController.showLock()
                    return EditStates.locked
                @cursorController.showRotationCursor(@origin, mouse)
                return EditStates.rotating

        @cursorController.remove()
        return EditStates.idle

    updateState: (newState) =>
        return if @state is newState
        @controllerForState()?.end?(@)
        @state = newState
        @controllerForState()?.start?(@)
        @viewController.render()

    controllerForState: =>
        switch @state
            when EditStates.moving          then return @translationTool
            when EditStates.resizing        then return @resizingTool
            when EditStates.rotating        then return @rotationTool
            when EditStates.transform       then return @transformController
            when EditStates.borderRadius    then return @borderRadiusTool
            when EditStates.animation       then return @animationTool
        return null

    exit: =>
        @updateState(EditStates.idle)
        @updateLayerInfo()
        @stateForPosition(@mouse)

    # layer info

    @define "origin",
        get: -> return @viewController.origin.centerPoint

    updateLayerInfo: =>
        return unless @layer?

        for handle in @resizeHandles
            handle.attachToLayer(@layer)
            handle.updatePosition()

        @viewController.borderRadiusHandle?.updatePosition()

        @viewController.render()
        if @state != EditStates.animation
          @snappingVisualizer.showLayerOutline(@layer)

    layerPolyline: =>
        return [@resizeHandles[0].point, @resizeHandles[6].point, @resizeHandles[8].point, @resizeHandles[2].point]

    centerPoint: =>
        return @resizeHandles[4].point

    editTarget: (target) ->
        @relatedLayers = ViewerTools.getLayersForTarget(target) if target?
        layer = _.first(@relatedLayers)
        if not layer?
            @stop()
            return false

        log "edit layer", layer
        @editLayer(layer)
        return true

    editLayer: (layer) ->

        return @updateLayerInfo() if @layer is layer

        window.oncontextmenu = -> return false

        Framer.Extras.TouchEmulator.disable()

        return if layer instanceof BackgroundLayer

        @layer = layer
        @propertyLock.attachToLayer(layer)
        @updateLayerInfo()

        @bodyCursor = document.body.style.cursor
        document.body.style.cursor = ""

        return true

    stop: ->

        log "stop editing layer"

        if @state isnt EditStates.idle
            @state = EditStates.idle
            bridge.send "drag:end"

        window.oncontextmenu = null

        @mouse = null
        @hash = null
        @layer = null
        @relatedLayers = null

        @snappingModel.end()
        @viewController.end()
        @snappingVisualizer.hideOutline()
        @tooltip.hide()
        @cursorController.remove()

        @animationTool.reset()

        document.body.style.cursor = @bodyCursor if @bodyCursor?

        Framer.Extras.TouchEmulator.enable()

        if Framer.CurrentContext.__framerViewFrozen
            Framer.CurrentContext.resume()
            delete Framer.CurrentContext.__framerViewFrozen

    layerChanged: (beforeChange) =>
        return unless @layer?

        properties = @propertyLock.activeProperties()

        for relatedLayer in @relatedLayers when @relatedLayers?
            for property in properties
                relatedLayer[property] = @layer[property]
        fromValues = _.pick(beforeChange, properties)
        toValues = _.pick(@layer, properties)
        @animationTool.updateProperties(fromValues, toValues)
        @sendThrottledToBridge()

    sendToBrige: =>
        properties = @propertyLock.activeProperties()
        bridge.send "framerView:layerEdited",
            id:             @layer.id
            hash:           @layer.__framerInstanceInfo?.hash
            properties:     _.pick(@layer, properties)
