EventEmitter			= require "eventemitter3"
{bridge} 				= require "./Bridge"
{ViewerTools}			= require "./tools/ViewerTools"

BUILDS = 0

parseUrl = (url) ->
	# https://gist.github.com/jlong/2428561
	parser = document.createElement("a")
	parser.href = url
	return parser

class Runtime extends EventEmitter

	constructor: ->
		super
		@setup()

	setup: ->

		# Listen for device changes and pass them on to the viewer so we
		# can update the menus in the toolbar
		if Framer?.Device
			properties = [ "deviceScale", "contentScale", "deviceType",
				"keyboard", "orientation", "fullScreen"]
			properties.map (propertyName) ->
				Framer.Device.on "change:#{propertyName}", ->
					bridge.send("device:change")

			Framer.Device.on "change:phoneScale", (phoneScale) ->
				bridge.send("change:phoneScale", {phoneScale: phoneScale})

			Framer.Device._calculatePhoneScale()

		bridge.responders["getAnimationOptions"] = (animationData) ->
			if not Framer.Curves?
				return null
			{layerHash, animationHash, animationOptions} = animationData
			layers = ViewerTools.getAllLayersWithHash(layerHash)
			animations = layers.map (layer) ->
				layer.__framerAnimationInfo?[animationHash]
			animations = animations.filter (animation) -> animation?
			firstAnimation = _.first(animations)
			if firstAnimation?
				options = firstAnimation.destinationOptions
			else if animationOptions.curve?
				options = animationOptions
				options.curve = Framer.Curves.fromString(animationOptions.curve)
				if options.curve == Framer.Curves.Spring
					options.curve = options.curve.call()
			else
				return null
			animator = options.curve(options)
			animatorOptions = animator.options
			if not animatorOptions.time? and animatorOptions.tension? and animatorOptions.friction?
				duration = Spring.computeDuration(animatorOptions.tension, animatorOptions.friction, animatorOptions.velocity, animatorOptions.mass)
				if duration?
					animatorOptions.time = duration
					animatorOptions.dampingRatio = Spring.computeDampingRatio(animatorOptions.tension, animatorOptions.friction, animatorOptions.mass)

			animatorName = animator.constructor.name
			switch animatorName
				when "BezierCurveAnimator"
					animatorOptions.curveType = "cubic-bezier"
					if options.curve.name? and options.curve.name.length > 0
						animatorOptions.timingFunction = _.kebabCase(options.curve.name)
					else if animatorOptions.values?
						animatorOptions.timingFunction = "cubic-bezier(#{animatorOptions.values.join(", ")})"
				when "SpringRK4Animator"
					animatorOptions.curveType = "spring"
				else
					animatorOptions.curveType = "unknown"
			return _.pick(animatorOptions, ["tension", "friction", "velocity", "dampingRatio", "mass", "delay", "time", "curveType", "timingFunction"])

		# Add a responder to the bridge, that provides layer properties:
		bridge.responders["layerProperties"] = (layerData) =>

			{ ownerHash, relatedHashes, relatedLookup, frozenMembers, panelType } = layerData

			relatedHashes.push(ownerHash)
			hashes = relatedHashes
			layers = ViewerTools.getLayersWithHashes(hashes)
			lookupLayers = ViewerTools.lookupLayers(relatedLookup)
			layers = layers.concat(lookupLayers)
			layer = _.first(layers)
			if not layer?
				console.log("WARNING: Layer #{ownerHash} not found!")
				return null

			unless Framer.CurrentContext.__framerViewFrozen
				for related in layers
					for animation in related.animations(true)
						if panelType is "propertiesPanel"
							# We're editing the final state of the animation
							animation.finish()
						else if panelType is "animationPanel"
							# We want to start an the inital state of the animation
							animation.stop()
							animation.reset()

				# Freeze the current context
				Framer.CurrentContext.freeze()
				Framer.CurrentContext.__framerViewFrozen = true

				# Hide the Print Context
				for context in Framer.Context.all()
					if context._name is "PrintConsole"
						context.visible = false

			# Now send out the selected layer properties:
			result = {}

			for key, value of layer
				if (value instanceof Color)
					result[key] = value.toRgbString()
				else if not (_.isArray(value) or _.isObject(value) or _.isFunction(value) or isNaN(value))
					result[key] = value
			layer.draggable.enabled = false

			return result

		# Tell Framer Studio we are ready
		bridge.send("runtime.init")

		# Set up the error handler
		@_errorHandlerSetup()

	reset: ->
		Framer.CurrentContext.emit("willReset")
		Utils.reset()
		Framer.Extras?.TouchEmulator?.enable()

	loaded: ->
		if Framer.CurrentContext.__framerViewFrozen
			Framer.CurrentContext.freeze()

	_errorHandler: (runtimeError) =>

		# This function catches runtime errors, so after the coffeescript was successfully
		# compiled without syntax errors. We have to detect if they come from the compiled
		# coffeescript or from any other library (most likely framer.js). The way to detect
		# it seems to be the .filename property, which is equal to the current location if
		# the error comes from the compiled coffeescript.

		errorFromCompiledCoffeeScript = (runtimeError.filename == window.location.href)

		error =
			message: runtimeError.message
			lineNumber: runtimeError.lineno
			colNumber: runtimeError.colno
			fileName: runtimeError.filename

		# If we get one of those secure generic errors, we tweak the messaging a bit.
		if error.message is "Script error."
			error.message = "Generic script error. See the inspector for more information."

		if errorFromCompiledCoffeeScript
			error.type = "coffeescript"
			error.clientCodeId = window.__framerClientCodeId

		else
			error.type = "javascript"
			fileName = _.last(parseUrl(runtimeError.filename).pathname.split("/"))

			# Only show the filename if this is from a known javascript file, if not
			# just show it as a generic, global script error.
			if _.endsWith(fileName, ".js")
				error.message = "[#{fileName}] #{error.message}"
			else
				error.message = "#{error.message}"

		bridge.send("runtime.error", error)

	_errorHandlerSetup: ->
		window.addEventListener("error", @_errorHandler)

	_errorHandlerRemove: ->
		window.removeEventListener("error", @_errorHandler)

exports.runtime = new Runtime()
