bridge              =(require "./../shared/Bridge").bridge
utils               = require "./../shared/Utils"
{Constants}         = require "./../shared/Constants"
{TableComponent}    = require "./TableComponent"


class exports.LayerPanel extends TableComponent

    constructor: (options = {}) ->

        options.name ?= "LayerPanel"

        @messages = Constants.messages.layerpanel
        @data = []
        super options
        @render()

        framerViewMessages = Constants.messages.framerview
        framerStudioMessages = Constants.messages.studio

        bridge.on framerViewMessages.layers, @setData
        bridge.on framerStudioMessages.state, @handleStudioState
        bridge.on framerStudioMessages.layerPanelEnter, @handleStudioLayerPanelEnter
        bridge.on framerStudioMessages.layerPanelLeave, @handleStudioLayerPanelLeave

        @bumper.on Events.MouseOver, =>
            if @_highlightedLayerId?
                @unhighlightRow()
                bridge.send(@messages.itemRollOut, {})

    setData: (data) =>

        console.log("layerPanel:setData")

        # We need to check if the current highlighted layer is still existing.
        # If not we need to reset the highlighter.
        if @_highlightedLayerId and @_highlightedLayerId not in _.pluck(data, "id")
            bridge.send(@messages.reset, {})

        # Sort the layers by hierarchy and render the table view again
        @data = utils.sortLayersByHierarchy(data)
        @render()

    totalRows: -> @data.length
    heightForRow: -> Constants.sizes.rowHeight
    marginForRow: -> 0

    createRow: ->
        row = new Layer
            height: @heightForRow()

        row.style =
            borderBottom: "1px solid #{Constants.colors.border}"

        # Indentation layer (for the hierarchy)
        row.indentLayer = new Layer
            x: Constants.sizes.indent
            width: row.width - Constants.sizes.indent
            height: Constants.sizes.indentHeight
            parent: row

        row.indentLayer.style =
            background: "repeating-linear-gradient(to right,
                    #{Constants.colors.indent},
                    #{Constants.colors.indent} #{Constants.sizes.indentWidth}px,
                    #{Constants.colors.background} #{Constants.sizes.indentWidth}px,
                    #{Constants.colors.background} #{Constants.sizes.indent}px
                )"
        row.indentLayer.centerY()

        # Text layer (containing the name)
        row.textLayer = new Layer
            width: row.width
            height: @heightForRow()
            parent: row
            backgroundColor: Constants.colors.background

        row.textLayer.style =
            font: "11px -apple-system, Helvetica Neue, Helvetica"
            color: Constants.colors.text
            lineHeight: "#{@heightForRow()}px"
            "-webkit-user-select": "none"
            "-webkit-font-smoothing": "subpixel-antialiased"

        # Events
        row.onMouseOver @handleMouseOver
        row.onMouseDown @handleClick
        row._element.addEventListener(
            "contextmenu",
            @handleContextMenu(row)
            false)
        row

    updateRow: (index, row) ->
        row.data = @data[index]
        row.textLayer.html = "#{row.data.name}"

        indent = (row.data.level + 1) * Constants.sizes.indent
        row.textLayer.x = indent
        row.textLayer.width = row.width - indent

        if @isRowSelected("hover", row)
            row.textLayer.style.color = Constants.colors.textHighlight
        else
            row.textLayer.style.color = @_textColor(row.data)

        color = if @isRowSelected("selection", row)
            Constants.colors.tableSelectionBackground
        else
            Constants.colors.background

        row.textLayer.backgroundColor = color
        row.backgroundColor = color

    handleMouseOver: (event, row) =>
        @_highlightedLayerId = row?.data?.id
        @sendItemRollOver row

    handleClick: (event, row) =>
        if event.metaKey
            bridge.send(@messages.itemCmdClick,
                hash: row.data.__framerInstanceInfo?.hash
                state: row.data.currentState
            )
        else
            bridge.send(@messages.itemClick,
                id: row.data.id
                data: row.data.__framerInstanceInfo
            )

    handleContextMenu: (row) =>
        return (event) =>
            bridge.send(@messages.itemContextMenu,
                id: row.data.id
                data: row.data.__framerInstanceInfo
                location: {x: event.pageX, y: event.pageY}
            )

    handleStudioState: (state) =>

        console.log "layerPanel:handleStudioState", state

        if @_highlightedLayerId == state.highlightingTarget?.runtimeId
            return

        id = state.highlightingTarget?.runtimeId
        if id?
            for row in @_rowLayers
                if row.data.id == id
                    @highlightRow row
                    return

        @unhighlightRow()

    highlightRow: (row) ->
        @setSelectedRows("hover", [row])
        @_highlightedLayerId = row.data.id

    unhighlightRow: ->
        @setSelectedRows("hover", [])
        @_highlightedLayerId = null

    handleScrollerResize: (width, height) =>
        @width = width

    handleStudioLayerPanelEnter: =>

        console.log("layerPanel:handleStudioLayerPanelEnter")

        @entered = true
        @delay = Utils.delay .25, =>
            clearTimeout @delay
            @delay = null
            @sendItemRollOver @delayedRow if @delayedRow?
            @delayedRow = null

    handleStudioLayerPanelLeave: =>

        console.log("layerPanel:handleStudioLayerPanelLeave")

        clearTimeout @delay if @delay?
        @delayedRow = null
        @delay = null
        @entered = false

    sendItemRollOver: (row) =>
        if @entered and not @delay?
            console.log "sentItemRollOver"
            @highlightRow row
            bridge.send(@messages.itemRollOver, {id:row.data.id, data: row.data.__framerInstanceInfo})
        else
            @delayedRow = row

    # Internals
    #

    _textColor: (data) ->
        return Constants.colors.text if data.visibleResult
        return Constants.colors.textInvisible

# Utils extension
#

Utils.convertPoint = (input, layerA, layerB, context=false) ->

    # Convert a point between two layer coordinate systems

    point = _.defaults(input, {x:0, y:0})

    superLayersA = layerA?.superLayers(context) or []
    superLayersB = layerB?.superLayers(context) or []

    superLayersB.push(layerB) if layerB

    for layer in superLayersA
        point.x += layer.x - layer.scrollFrame.x
        point.y += layer.y - layer.scrollFrame.y

    for layer in superLayersB
        point.x -= layer.x + layer.scrollFrame.x
        point.y -= layer.y + layer.scrollFrame.y

    return point
