# coding=utf-8

import os
import sys
import logging
import mimetypes
import json
import re
import itertools
import threading
import time
import socket
import traceback
import subprocess
import sys
import Queue
import string
import random
import urllib
import hashlib
import defaults
import netifaces
import utils

logger = logging.getLogger(__name__)

import tornado.httpserver
import tornado.websocket
import tornado.websocket
import tornado.template
import tornado.ioloop
import tornado.web

from tornado.escape import json_encode, json_decode, to_unicode, url_escape

HTTPError = tornado.web.HTTPError

class APIResponseError(Exception):
	pass

defaults = defaults.Defaults("com.motif.framer.server")

def is_subdir(suspect_child, suspect_parent):
	suspect_child = os.path.realpath(suspect_child)
	suspect_parent = os.path.realpath(suspect_parent)

	relative = os.path.relpath(suspect_child, start=suspect_parent)

	return not relative.startswith(os.pardir)

def path_in_current_user_directory(path):
	return is_subdir(path, os.path.expanduser("~"))

def random_string(length):
	return os.urandom(length).encode("hex")[:length]

def random_number(length):
	return str(int(random_string(length), 16))[:length]

class WebSocketHandler(tornado.websocket.WebSocketHandler):

	def open(self):
		if self not in self.application._socket_handlers:
			self.application._socket_handlers.append(self)

	def on_close(self):
		if self in self.application._socket_handlers:
			self.application._socket_handlers.remove(self)

	def on_message(self, msg):
		pass


class StaticFileHandler(tornado.web.StaticFileHandler):
	"""
	Unsecure static file handler for static assets.
	"""

	def get_content_type(self):

		root, ext = os.path.splitext(self.absolute_path)

		extra_content_types = {
			".coffee": "text/coffeescript"
		}

		if extra_content_types.get(ext, None):
			mime_type = extra_content_types[ext]
		else:
			mime_type, encoding = mimetypes.guess_type(self.absolute_path)

		return mime_type

	def is_text_content(self):

		mime_type = self.get_content_type()

		if not mime_type:
			return False

		for text_type in ["text", "javascript", "json"]:
			if text_type in mime_type:
				return True

		return False

	def write_error(self, status_code, **kwargs):

		loader = tornado.template.Loader(os.path.join(self.application.static_path, "error"))

		if status_code == 404:
			return self.finish(loader.load("404.html").generate())

		return super(StaticFileHandler, self).write_error(status_code, **kwargs)


class BaseProjectHandler(tornado.web.RequestHandler):

	def initialize(self, **kwargs):
		self.mount = kwargs["mount"]
		del kwargs["mount"]
		self.secure_cookie_user_key = u"framer|{0}".format(self.mount["handle"])
		super(BaseProjectHandler, self).initialize(**kwargs)

	def get_current_user(self):
		return self.get_secure_cookie(self.secure_cookie_user_key)

	def logout(self):
		self.clear_cookie(self.secure_cookie_user_key, path=self.get_project_url())

	def get_login_url(self):
		return u"/{0}/login".format(self.mount["handle"])

	def get_project_url(self):
		return self.mount["url"]

class ProjectLoginHandler(BaseProjectHandler):

	def get(self):

		loader = tornado.template.Loader(
			os.path.join(self.application.static_path, "login"))

		return self.finish(loader.load("index.html").generate())

	utils.rate_limited(2)
	def post(self):

		code = self.mount["code"]
		salt = self.get_argument("salt")

		# Make sure we have a decent sized salt value
		if len(salt) < 128:
			return self.write({"error": "Salt value too small"})

		# Make sure the salt value is random and has not been used before
		if not hasattr(self, "_recently_used_salts"):
			self._recently_used_salts = []

		if salt in self._recently_used_salts:
			return self.write({"error": "Salt value already used."})

		# Caclculate and compare the hashes to see if the password is correct
		client_hash = self.get_argument("hash")
		server_hash = hashlib.sha256("" + salt + code).hexdigest()

		# Compare hashes to see if we're ok
		if client_hash != server_hash:
			return self.write({"error": "Code does not match."})

		# TODO: Should we add the ip address here and check it later for extra security and
		# to avoid session hijacking?
		self.set_secure_cookie(self.secure_cookie_user_key, "framer", path=self.get_project_url())

		self.write({
			"error": None,
			"redirect": self.get_project_url()
		})

class StaticProjectFileHandler(BaseProjectHandler, StaticFileHandler):
	"""
	Secure static file handler for assets under /<project/*. This handler automatically
	adds the server.js script, disables caching and adds cookie based security.
	"""

	@classmethod
	def get_append(cls, abspath):

		mime_type, encoding = mimetypes.guess_type(abspath)

		if mime_type == "text/html":
			return u"\n<script src=\"/_server/websocket/server.js\"></script>"

		return u""

	@classmethod
	def get_content(cls, abspath, start=None, end=None):
		content = super(StaticProjectFileHandler, cls).get_content(abspath, start=start, end=end)
		content = itertools.chain(content, [cls.get_append(abspath)])
		return content

	def get(self, *args, **kwargs):

		# These are special urls that can be seen even without logging in
		whitelist = [
			u"{0}framer/preview.png".format(self.get_project_url()),
		]

		# If we hit a whitelisted path, we just return without authentication
		if self.request.path in whitelist:
			return super(StaticProjectFileHandler, self).get(*args, **kwargs)

		# See if we need to do a force login/logout
		notset = "notset"

		logout = self.get_argument("logout", notset)
		login  = self.get_argument("login", notset)

		if logout != notset or login != notset:
			self.logout()

		if login != notset:
			return self.redirect(self.get_login_url())

		# Only check authentication if this is a non-local request
		if not self.application.is_local_request(self.request):

			# If we are not authenticated yet
			if not self.current_user:

				# Only redirect if this is an html request
				is_api_request = "json" in self.request.headers.get("accept", "")
				is_get_request = self.request.method in ("GET", "HEAD")
				is_html_request = self.request.path.endswith("/") or self.request.path.endswith(".html")

				if is_get_request and is_html_request and not is_api_request:
					return self.redirect(self.get_login_url())
				
				# If not, just raise an error
				raise HTTPError(401, reason="401 Unauthorized")

		# If authenticated or a local request we return the content
		return super(StaticProjectFileHandler, self).get(*args, **kwargs)

	def get_content_size(self):
		return super(StaticProjectFileHandler, self).get_content_size() + \
			len(self.get_append(self.absolute_path))

	def set_extra_headers(self, path):
		if self.is_text_content():
			self.set_header("Cache-control", "no-cache")

	def should_return_304(self):

		if self.is_text_content():
			return False

		return super(StaticFileHandler, self).should_return_304()

	# def validate_absolute_path(self, root, absolute_path):

	# 	if not path_in_current_user_directory(absolute_path):
	# 		raise HTTPError(403, "%s is not in current user directory", absolute_path)

	# 	return super(StaticFileHandler, self).validate_absolute_path(root, absolute_path)


class APIHandler(tornado.web.RequestHandler):

	def get(self, command):

		command_name = "do_{0}".format(command)

		if hasattr(self, command_name):
			self.set_header("Content-Type", "application/json; charset=utf-8")

			try:
				body = {u"result": getattr(self, command_name)()}
				self.finish(json.dumps(body, ensure_ascii=False))
			except Exception, e:

				logging.error(traceback.format_exc())

				# This is mainly for the tests
				if not isinstance(e, APIResponseError):
					print traceback.format_exc()

				self.set_status(500)
				self.finish(json_encode({"error": e.message}))

		else:
			self.set_status(400)
			self.finish("404 Not found")

	def post(self, command):
		self._check_security() # Better safe then sorry
		self.get(command)

	def do_list(self):

		data = []

		for k, v in self.application._mounts.iteritems():

			mount = {
				"handle": v["handle"],
				"path": v["path"],
				"url": v["url"],
				"name": v["name"],
				"saved": self._is_saved_project(v["path"])
			}

			data.append(mount)

		return data

	def do_mount(self):
		self._check_security()
		
		path = self._get_argument("path")
		path = os.path.abspath(path)

		if not os.path.exists(path):
			raise APIResponseError("Path does not exist %s" % path)

		return self.application.mount(path)

	def do_unmount(self):
		self._check_security()

		path = self._get_argument("path")
		path = os.path.abspath(path)

		if not os.path.exists(path):
			raise APIResponseError("Path does not exist %s" % path)

		self.application.unmount(path)

		return "OK"

	def do_eval(self):
		self._check_security()
		script = self._get_argument("script")
		self.application.eval(script)
		return "OK"

	def do_ping(self):
		self._check_security()
		return "OK"

	def _get_argument(self, key, required=True):

		data = json_decode(self.request.body)

		if data.get(key, None) is None:
			raise Exception("Required key missing %s" % key)

		return data[key]

	def _check_security(self):
		# Only the machine this is running on is allowed
		if not self.application.is_local_request(self.request):
			raise HTTPError(403, "{0} is not a local ip".format(self.request.remote_ip))

	def _is_saved_project(self, path):
		# This is a quick hack that relies on the project being in a temp folder
		return not path.startswith("/var")

class FramerServerApp(tornado.web.Application):

	def __init__(self, port=8080, debug=False):

		# self.path = path
		self.port = port
		self.static_path = os.path.join(os.path.dirname(__file__).decode('utf-8'), u"www")
		self.app_resource_path = os.environ.get("BUNDLE_RESOURCE_PATH", "").decode('utf-8')

		# print "self.app_resource_path", self.app_resource_path

		self.local_ip = u"127.0.0.1"

		self._url = u"http://{0}:{1}".format(self.local_ip, self.port).lower()
		self._socket_handlers = []
		self._mounts = {}
		self._handlers = {}

		cookie_secret = defaults.get("cookie_secret")

		if not cookie_secret:
			cookie_secret = random_string(128)
			defaults.set("cookie_secret", cookie_secret)

		super(FramerServerApp, self).__init__([
			tornado.web.url(ur"/_server/ws", WebSocketHandler),
			tornado.web.url(ur"/_server/api/(.*)", APIHandler),
			tornado.web.url(ur"/_server/resources/(.*)", StaticFileHandler, {
				"path": self.app_resource_path,
				"default_filename": u"index.html"}),
			tornado.web.url(ur"/_server/(.*)", StaticFileHandler, {
				"path": self.static_path,
				"default_filename": u"index.html"}),
			tornado.web.url(ur"/", tornado.web.RedirectHandler, {"url": u"/_server/app"}),
		],
		debug=debug,
		serve_traceback=debug,
		autoreload=False,
		cookie_secret=cookie_secret,
		login_url="")

	@property
	def url(self):
		return self._url

	# Utility methods

	@utils.memoize(timeout=60)
	def _local_adresses(self):
		return utils.get_local_adresses()

	def is_local_request(self, request):
		return request.remote_ip in self._local_adresses()

	def _get_normalized_path(self, path):
		return os.path.abspath(path)

	def _get_name_for_path(self, path):
		return os.path.basename(path)

	# Server methods

	def start(self):
		self._server = tornado.httpserver.HTTPServer(self)
		self._server.listen(self.port, "0.0.0.0")

		tornado.ioloop.IOLoop.instance().start()

	def stop(self):
		for handle, mount in self._mounts:
			self.unmount(mount["path"])

	# Mount and unmount

	def mount(self, path):

		path = self._get_normalized_path(path)
		current_mount = self._get_mount_for_path(path)

		if current_mount:
			logging.info("Path already mounted: %s", path)
			return current_mount

		name = self._get_name_for_path(path)
		name = name.replace("(", "")
		name = name.replace(")", "")

		handle = hashlib.sha1(path.encode("utf8")).hexdigest()[:6]
		code = random_number(6)		

		# if not path_in_current_user_directory(path):
		# 	error = "reloader.mount: path not in user home path: {0}".format(path)
		# 	logging.warning(error)
		# 	raise Exception(error)

		if not os.path.exists(path):
			error = "reloader.mount: no such path: {0}".format(path)
			logging.warning(error)
			raise APIResponseError(error)

		mount = {
			"handle": handle,
			"name": name,
			"path": path,
			"url": u"/{0}/{1}/".format(handle, url_escape(name)),
			"code": code
		}

		self._mounts[handle] = mount


		handlers = [
			(r"/{0}/login".format(handle), ProjectLoginHandler, {
				"mount": mount,
			}, "{0}-login".format(mount["handle"])),
			(r"/{0}/{1}/?(.*)".format(handle, re.escape(url_escape(name))), StaticProjectFileHandler, {
				"path": path,
				"default_filename": u"index.html",
				"mount": mount,
			}, "{0}-project".format(mount["handle"])),
			# Very temporary hack for double encoding on iOS
			(r"/{0}/{1}/?(.*)".format(handle, re.escape(urllib.quote(url_escape(name)))), StaticProjectFileHandler, {
				"path": path,
				"default_filename": u"index.html",
				"mount": mount,
			}, "{0}-project".format(mount["handle"])),
			(r"/{0}/{1}/?(.*)".format(handle, re.escape(url_escape(urllib.quote(name.encode('ascii', 'ignore'))))), StaticProjectFileHandler, {
				"path": path,
				"default_filename": u"index.html",
				"mount": mount,
			}, "{0}-project".format(mount["handle"])),
		]

		self.add_handlers(ur".*", handlers)
		self._handlers[mount["handle"]] = handlers

		return mount

	def unmount(self, path):

		mount = self._get_mount_for_path(path)
		handle = mount["handle"]

		# for name, handler in self.named_handlers.iteritems():
		# 	if name.startswith(handle):
		# 		self.handlers.remove(handler)

		for handler in self.handlers:
			for spec in handler[1]:
				if getattr(spec.kwargs, "mount", None) == mount:
					self.handlers.remove(handler)
					continue

		del self._mounts[handle]

	def _get_handler_for_handle(self, handle):
		for handler in self.handlers:
			if handle == handler[1][0].name:
				return handler

	def _get_mount_for_path(self, path):
		for handle, m in self._mounts.iteritems():
			if m["path"] == path:
				return m

	# Websocket methods

	def publish(self, message):
		for ws in self._socket_handlers:
			ws.write_message(message)

	def eval(self, script):
		self.publish({"command": "eval", "script": script})
