{Line, Point} = require "./Line"

class Snap

	constructor: (layer, value, coord) ->
		@value = new Number(value)
		@layer = layer
		@coord = coord

class SnapPicker

	constructor: ->
		@reset()
	
	add: (snap) ->
		value = snap.value
		@map["#{value}"] = [] unless @map["#{value}"]
		valueArray = @map["#{value}"]
		valueArray.push(snap)
		count = valueArray.length
		if count > @_bestSnapValueCount
			@_bestSnapValueCount = count
			@_bestSnapValue = value
	
	best: ->
		if @_bestSnapValue == null
			return false
		return @map["#{@_bestSnapValue}"]

	coordinates: ->
		result = []
		if @best()
			for snap in @best()
				result.push(snap.coord)
		return result
	
	reset: ->
		@map = new Object
		@_bestSnapValue = null
		@_bestSnapValueCount = 0

class exports.SnappingModel

	constructor: ->
		@horizontalSnapPicker = new SnapPicker
		@verticalSnapPicker = new SnapPicker

	reset: ->
		@horizontalSnapPicker.reset()
		@verticalSnapPicker.reset()

	end: =>
		@_siblingCoords = null
		@_siblingCoordsExcludingDescendants = null

		@_siblingLines = null
		@_siblingLinesExcludingDescendants = null

	moveHandleWithConstraints: (layer, relatedLayers, point, handle, constraints) =>

		@reset()

		snapResult =
			point: point

		snapThreshold = 12
		siblingLines = @siblingLines(layer, relatedLayers)
		{fx, fy} = handle

		constraintsA = Utils.convertPointFromContext(handle.point, layer, true, false)
		constraintsB = Utils.convertPointFromContext(constraints, layer, true, false)

		withoutX = constraintsA.x is constraintsB.x
		withoutY = constraintsA.y is constraintsB.y

		constraintLine = new Line(constraintsA, constraintsB)

		intersections = []

		for sibLine in siblingLines

			intersection = constraintLine.intersection(sibLine)

			unless intersection?
				continue

			if withoutX
				if Math.abs(intersection.y - point.y) < snapThreshold
					snapResult.point = intersection
					snapResult.line = sibLine
					break
			else if withoutY
				if Math.abs(intersection.x - point.x) < snapThreshold
					snapResult.point = intersection
					snapResult.line = sibLine
					break
			else if intersection.distance(point) < snapThreshold
				snapResult.point = intersection
				snapResult.line = sibLine
				break

		return snapResult

	moveHandle: (layer, relatedLayers, point, handle) =>

		@reset()

		snapThreshold = 12
		siblingPoints = @siblingCoords(layer, relatedLayers)
		{fx, fy} = handle

		for sibPoint in siblingPoints

			xOffset = point.x - sibPoint.x
			yOffset = point.y - sibPoint.y

			if Math.abs(xOffset) < snapThreshold and fx isnt 0.5
				xValue = @round(point.x - xOffset)
				snap = new Snap(layer, xValue, sibPoint)
				@horizontalSnapPicker.add(snap)

			if Math.abs(yOffset) < snapThreshold and fy isnt 0.5
				yValue = @round(point.y - yOffset)
				snap = new Snap(layer, yValue, sibPoint)
				@verticalSnapPicker.add(snap)

		if @horizontalSnapPicker.best()
			snap = @horizontalSnapPicker.best()[0]
			point.x = snap.value

		if @verticalSnapPicker.best()
			snap = @verticalSnapPicker.best()[0]
			point.y = snap.value

		snapResult =
			point: point
			horizontalCoordinates: @horizontalSnapPicker.coordinates()
			verticalCoordinates: @verticalSnapPicker.coordinates()

		return snapResult

	moveLayer: (layer, relatedLayers, point) =>

		@reset()

		snapThreshold = 12

		layer.x = point.x
		layer.y = point.y

		pointsInSuper = @pointsInSuper(layer)
		siblingPoints = @siblingCoords(layer, relatedLayers, false)

		for lPoint in pointsInSuper
			for sibPoint in siblingPoints

				xOffset = lPoint.x - sibPoint.x
				yOffset = lPoint.y - sibPoint.y

				if Math.abs(xOffset) < snapThreshold
					xValue = @round(point.x - xOffset)
					snap = new Snap(layer, xValue, sibPoint)
					@horizontalSnapPicker.add(snap)

				if Math.abs(yOffset) < snapThreshold
					yValue = @round(point.y - yOffset)
					snap = new Snap(layer, yValue, sibPoint)
					@verticalSnapPicker.add(snap)

		if @horizontalSnapPicker.best()
			snap = @horizontalSnapPicker.best()[0]
			point.x = snap.value
			layer.x = snap.value

		if @verticalSnapPicker.best()
			snap = @verticalSnapPicker.best()[0]
			point.y = snap.value
			layer.y = snap.value

		snapResult =
			point: point
			horizontalCoordinates: @horizontalSnapPicker.coordinates()
			verticalCoordinates: @verticalSnapPicker.coordinates()

		return snapResult

	siblingCoords: (layer, relatedLayers, includeDescendants = true) =>

		result = []

		if includeDescendants and @_siblingCoords?
			result = @_siblingCoords
		else if !includeDescendants and @_siblingCoordsExcludingDescendants?
			result = @_siblingCoordsExcludingDescendants
		else

			descendants = layer.descendants

			for contextLayer in Framer.CurrentContext._layers

				continue if contextLayer is layer or (relatedLayers? and contextLayer in relatedLayers)
				continue if !includeDescendants and contextLayer in descendants
				points = @pointsToLayer(contextLayer, layer)
				result.push(points...)

			@_siblingCoords = result if includeDescendants
			@_siblingCoordsExcludingDescendants = result if !includeDescendants

		screenCoords = @screenCoords(layer)
		return result.concat(screenCoords)

	siblingLines: (layer, relatedLayers, includeDescendants = true) =>

		result = []

		if includeDescendants and @_siblingLines?
			result = @_siblingLines
		else if !includeDescendants and @_siblingLinesExcludingDescendants?
			result = @_siblingLinesExcludingDescendants 
		else
			descendants = layer.descendants

			for contextLayer in Framer.CurrentContext._layers

				continue if contextLayer is layer or (relatedLayers? and contextLayer in relatedLayers)
				continue if !includeDescendants and contextLayer in descendants
				lines = @linesToLayer(contextLayer, layer)
				result.push(lines...)

			@_siblingLines = result if includeDescendants
			@_siblingLinesExcludingDescendants = result if !includeDescendants

		screenLines = @screenLines(layer)
		return result.concat(screenLines)

	coords =
		topLeft:		x:0, y:0
		topRight:		x:1, y:0
		bottomLeft: 	x:0, y:1
		bottomRight: 	x:1, y:1
		center: 		x:0.5, y:0.5

	screenCoords: (layer) =>
		context = layer.context
		layerPoints = []

		for x in [0, .5, 1]
			for y in [0, .5, 1]
				xPoint = Math.round(x * context.width)
				yPoint = Math.round(y * context.height)
				contextPoint = {x:xPoint, y:yPoint}
				point = Utils.convertPointFromContext(contextPoint, layer, false, false)
				layerPoints.push(point)
		return layerPoints

	screenLines: (layer) =>
		points = @screenCoords(layer)
		return @linesFromNinePoints(points)

	pointsInSuper: (layer) =>

		layerPoints = []

		for key, coordinate of coords
			point = Utils.convertPoint({x:coordinate.x * layer.width, y:coordinate.y * layer.height}, layer, layer.superLayer)
			layerPoints.push(point)

		return layerPoints

	pointsToLayer: (contextLayer, layer) =>

		layerPoints = []
		for key, coordinate of coords
			# Convert a point between two layer coordinate systems
			point = Utils.convertPointToContext({x:coordinate.x * contextLayer.width, y:coordinate.y * contextLayer.height}, contextLayer, false)
			layerPoints.push(Utils.convertPointFromContext(point, layer, false, false))
		return layerPoints

	ninePointsToLayer: (contextLayer, layer) =>
		points = []
		for x in [0, .5, 1]
			for y in [0, .5, 1]
				point = Utils.convertPointToContext({x:x * contextLayer.width, y:y * contextLayer.height}, contextLayer, false)
				points.push(Utils.convertPointFromContext(point, layer, false, false))
		return points

	linesFromNinePoints: (points) =>
		lineA = new Line(points[0], points[2])
		lineB = new Line(points[3], points[5])
		lineC = new Line(points[6], points[8])
		lineD = new Line(points[0], points[6])
		lineE = new Line(points[1], points[7])
		lineF = new Line(points[2], points[8])
		return [lineA, lineB, lineC, lineD, lineE, lineF]

	linesToLayer: (contextLayer, layer) =>

		points = @ninePointsToLayer(contextLayer, layer)
		return @linesFromNinePoints(points)

	round: (value, decimals = 2) =>
		precision = 10 * decimals
		return Math.round(value * precision) / precision


